# -*- coding: iso-8859-1 -*-
#
# quakepy/pmc/PMC.py
# $Id: PMC.py 186 2009-03-31 22:23:33Z fab $
#
# The QuakePy package
# http://www.quakepy.org
#

############################################################################
#    Copyright (C) 2007-2009 by Fabian Euchner and Danijel Schorlemmer     #
#    fabian@fabian-euchner.de                                              #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

"""
The QuakePy package
http://www.quakepy.org
"""

__version__  = '$Id: PMC.py 186 2009-03-31 22:23:33Z fab $'
__revision__ = '$Revision: 186 $'
__author__   = "Danijel Schorlemmer <ds@usc.edu>, Fabian Euchner <fabian@fabian-euchner.de>"
__license__  = "GPL"

import sys
import copy
import cPickle
import math
import numpy

import cStringIO
import pyRXP
from   xml.sax   import saxutils

# import matplotlib
# matplotlib.use('PS')
# from pylab import *

from mx.DateTime     import Date
from mx.DateTime.ISO import ParseDateTimeUTC

sys.path.append('..')
sys.path.append('.')

from QPCore                   import *
from QPAnnotation             import *
from QPColorMap               import *
from PMCInventory             import *
from PMCData                  import *
from WaveformStreamID         import WaveformStreamID

POS_TAGNAME, POS_ATTRS, POS_CHILDREN = range(3)

class PMC( QPObject ):
    """
    QuakePy: PMC
    """
    PICKINFO_COL_MAGNITUDE = 1
    PICKINFO_COL_DISTANCE  = 0
    PICKINFO_COL_PICKED    = 2

    # re-define these in any class derived from PMC
    stationsRequiredForTriggering = 4
    probThreshold                 = 0.999
    
    def __init__( self, stationlist_filename = None, dist_style = None, **kwargs ):
        """
        kwargs: encoding = 'ascii' | 'xml'
                smooth   = True | false
                combinations = Combinations object
        """
        if ( 'encoding' in kwargs.keys() ) and ( kwargs['encoding'] is not None ):
            encoding = kwargs['encoding']
        else:
            encoding = 'ascii'

        if ( 'smooth' in kwargs.keys() ) and ( kwargs['smooth'] is False ):
            self.distSmoothing = False
        else:
            self.distSmoothing = True

        if ( 'combinations' in kwargs.keys() ) and ( kwargs['combinations'] is not None ):
            self.combinations = kwargs['combinations']
        else:
            self.combinations = None
            
        self.distStyle = dist_style
        
        self.stations       = []
        self.stationAliases = []
        
        self.annotation = QPAnnotation()
        self.initAnnotation()

        if stationlist_filename is not None:
            self.importStations( stationlist_filename, encoding )

    def merge( self, T ):
        """
        merge contents of another PMC object with self
        """
        self.stations.extend( T.stations )
        self.stationAliases.extend( T.stationAliases )
        
    def save( self, filename ):
        fh = writeQPData( filename, binary = True )
        try:
            cPickle.dump( self, fh, 2 )
        except cPickle.PickleError:
            raise cPickle.PickleError, "PMC::save - error pickling PMC"
        fh.close()

    # -------------------------------------------------------------------------
    
    def readXML( self, stream ):

        # get XML file locally or from web
        # lines = getQPDataSource( filename, **kwargs ).read()

        # get whole content from stream
        lines = stream.read()

        # check if it is XML
        if not lines.startswith('<?xml'):
            raise IOError, 'PMC::readXML - input stream is not XML'
            
        tree = pyRXP.Parser().parse( lines )
        
        if tree[POS_TAGNAME] != 'PMC':
            raise TypeError, 'PMC::readXML - input stream is not PMC XML'

        self.fromXML( tree )
            
    def writeXML( self, output, **kwargs ):
        """
        serialize PMC object to XML

        kwargs:
            prettyPrint - pretty formatting of output XML
        """
        if isinstance( output, basestring ):
            ostream = writeQPData( output, **kwargs )
        else:
            ostream = output

        if 'prettyPrint' in kwargs.keys() and kwargs['prettyPrint'] is False:
            prettyPrint = False
        else:
            prettyPrint = True
            
        if prettyPrint is True:

            # serialize to string stream
            try:
                curr_stream = cStringIO.StringIO()
                self.toXML( 'PMC', curr_stream )
                streamSuccess = True
            except:
                streamSuccess = False
                print "PMC::writeXML - error in StringIO self.toXML()"

            if streamSuccess is True:
                try:
                    xmlPrettyPrint( curr_stream, ostream )
                    return
                except:
                    print "PMC::writeXML - error in xmlPrettyPrint()"

        # write to output stream w/o pretty print
        # fallback if prettify has not succeeded
        try:
            self.toXML( 'PMC', ostream )
        except:
            raise IOError, "PMC::writeXML - error in self.toXML()"

    # -------------------------------------------------------------------------
    
    def fromXML( self, tree, additionalElements = None ):
        
        # XML attributes
        if tree[POS_ATTRS]:
            attr_dict = tree[POS_ATTRS]
            if 'publicID' in attr_dict.keys():
                self.publicID = attr_dict['publicID']
                
        for child in tree[POS_CHILDREN]:
            
            # multiple children of same type
            if child[POS_TAGNAME] == 'PMCStation':
                sta = PMCStation( self.distStyle, smooth = self.distSmoothing )
                sta.fromXML( child, additionalElements )
                self.stations.append( sta )

            elif child[POS_TAGNAME] == 'stationAliases':

                # get values from attributes of stationAlias elements
                for newchild in child[POS_CHILDREN]:
                    if newchild[POS_TAGNAME] == 'stationAlias':

                        # stationAlias element has 3 required attributes
                        attr_dict = newchild[POS_ATTRS]
                        self.stationAliases.append( [ attr_dict['networkCode'], attr_dict['alias'], attr_dict['stationCode'] ] )

    def toXML( self, tagname, stream ):

        stream.write( '<?xml version="1.0" encoding="utf-8"?>' )
        stream.writelines( [ '<', tagname, ' xmlns="http://quakepy.org/xmlns/pmc/1.0">' ] )

        # multiple children of same type
        if hasattr( self, 'stations' ) and self.stations is not None:
            for sta in self.stations:
                sta.toXML( 'PMCStation', stream )
                
        if hasattr( self, 'stationAliases' ) and self.stationAliases is not None:

            if len( self.stationAliases ) > 0:

                stream.write( '<stationAliases>' )

                for sta_alias in self.stationAliases:
                    stream.write( ''.join ( ( '<stationAlias networkCode="', sta_alias[0],
                                              '" alias="', sta_alias[1],
                                              '" stationCode="', sta_alias[2], '"/>' ) ) )
                stream.write( '</stationAliases>' )

        stream.writelines( [ '</', tagname, '>' ] )
        return True

    ## -----------------------------------------------------------------------
        
    def assignPicks( self, catalog, **kwargs ):
        """
        kwargs: reduce_catalog = True - delete already assigned picks in catalog
                verbose_level  = 0 - no information
                                 1 - short information
                                 2 - detailed information
        """
        if catalog is None:
            raise ValueError, "PMC::assignPicks - no valid catalog given"

        if not isinstance( catalog, QPCatalog ):
            raise ValueError, "PMC::assignPicks - illegal Python type for earthquake catalog"
        
        if len( self.stations ) == 0:
            raise ValueError, "PMC::assignPicks - no station list loaded"

        if 'verbose_level' in kwargs.keys():
            verbose_level = kwargs['verbose_level']
        else:
            verbose_level = None

        if verbose_level is not None and verbose_level > 0:
            print " assign picks: number of events = %s" % catalog.size()

        # over stations
        for curr_station_idx, curr_station in enumerate( self.stations ):

            pickinfo_cols = self.stations[curr_station_idx].distribution.pickInfoCols
            
            if self._stationToBeUsed( curr_station ):
                
                # over events
                picks_used = 0

                # look if we append to an existing pickInfo
                if self.stations[curr_station_idx].distribution.pickInfo is not None:
                    # resize existing pickInfo, add as many rows as there are events

                    # get number of rows in existing pickInfo, add no. of events in current catalog
                    old_rows = self.stations[curr_station_idx].distribution.pickInfo.shape[0]
                    new_rows = old_rows + len(catalog.eventParameters.event)
                    self.stations[curr_station_idx].distribution.pickInfo.resize( (new_rows, pickinfo_cols), refcheck = 0 )
                else:
                    # create a new pickInfo
                    old_rows = 0
                    new_rows = len(catalog.eventParameters.event)
                    self.stations[curr_station_idx].distribution.pickInfo = numpy.zeros( (new_rows, pickinfo_cols), dtype=float )

                # only used well-behaved events (ensure this in preprocessCatalog)
                for curr_event_idx, curr_event in enumerate( catalog.eventParameters.event ):

                    # check if event is in one of the stations' "on" time windows
                    if self._eventDuringOntime( curr_event, curr_station ):

                        curr_ori = curr_event.getPreferredOrigin()

                        distanceXYZ, distanceXY = distanceBetweenPoints( ( curr_ori.latitude.value,
                                                                        curr_ori.longitude.value,
                                                                        curr_ori.depth.value ),
                                                                        ( curr_station.latitude,
                                                                        curr_station.longitude,
                                                                        -(curr_station.elevation/1000.0) ),
                                                                        2 )

                        magnitude = curr_event.getPreferredMagnitude()
                        if not magnitude:
                            raise ValueError, "PMC::assignPicks - event has no preferred magnitude"

                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 0] = float( distanceXYZ )
                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 1] = float( magnitude.mag.value )
                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 2] = float( self._eventPicked(curr_event, curr_station.networkCode, curr_station.stationCode) )
                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 3] = float( curr_event_idx )
                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 4] = float( self._calcMagnitudeFromDistance(distanceXYZ) )

                        # put focal time as decimal year in last column of pickInfo
                        self.stations[curr_station_idx].distribution.pickInfo[old_rows + picks_used, 5] = float( curr_ori.time.value.toDecimalYear() )

                        picks_used = picks_used + 1

                    # delete all picks that belong to the current station
                    if ( 'reduce_catalog' in kwargs.keys() and kwargs['reduce_catalog'] == True ):
                        self._deletePicks( curr_event, curr_station.networkCode, curr_station.stationCode )

                # resize pickInfo
                self.stations[curr_station_idx].distribution.pickInfo.resize( ( old_rows + picks_used,
                                                                                pickinfo_cols ),
                                                                                refcheck = 0 )

                if verbose_level is not None and ( verbose_level > 1 ) and ( picks_used > 1 ):

                    try:
                        locCode = curr_station.locationCode
                    except:
                        locCode = ''
                        
                    print " assigned station no. %s, %s %s %s, with %s picks / total pickInfo size is %s" % \
                        ( curr_station_idx, curr_station.networkCode, locCode, curr_station.stationCode,
                          picks_used, old_rows + picks_used )

    def importStations( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True

    def preprocess( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def preprocessInventory( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def preprocessCatalog( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def postprocess( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def importAliases( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def computePMC( self ):
        # uses methods that have been redefined in derived classes
        return True
    
    def initAnnotation( self ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def plotRawDistribution( self, station, metadata, imgfile = None, **kwargs ):
        """

        default colors for plotting events:
          not picked: grey
          picked:     black
          
        kwargs:
            imgformat: 'png' - convert to png, 'eps' no conversion
            plot_missed_color: Matplotlib color as string specification
            plot_picked_color: Matplotlib color as string specification
        """
        imgformat  = 'png'

        plot_missed_color = '0.6'
        plot_picked_color = 'k'
        
        if 'plot_missed_color' in kwargs.keys() and kwargs['plot_missed_color'] is not None:
            plot_missed_color = kwargs['plot_missed_color']

        if 'plot_picked_color' in kwargs.keys() and kwargs['plot_picked_color'] is not None:
            plot_picked_color = kwargs['plot_picked_color']

        ax = subplot(111)
        
        abscissa_picked    = []
        abscissa_notpicked = []
        ordinate_picked    = []
        ordinate_notpicked = []
        
        for curr_pick in station.distribution.pickInfo:
            if curr_pick[self.PICKINFO_COL_PICKED]:
                abscissa_picked.append( curr_pick[self.PICKINFO_COL_MAGNITUDE] )
                ordinate_picked.append( curr_pick[self.PICKINFO_COL_DISTANCE] )
            else:
                abscissa_notpicked.append( curr_pick[self.PICKINFO_COL_MAGNITUDE] )
                ordinate_notpicked.append( curr_pick[self.PICKINFO_COL_DISTANCE] )

        plot( abscissa_notpicked, ordinate_notpicked, color = plot_missed_color, marker = '.',
              linestyle = 'None', zorder = 1 )
        plot( abscissa_picked, ordinate_picked, color = plot_picked_color, marker = '.',
              linestyle = 'None', zorder = 2 )
        
        xlabel( 'Magnitude' )
        ylabel( 'Distance / km' )
        title( "Network " + station.networkCode + ", Station " + station.stationCode )
        
        # get filename
        if imgfile is None:

            # get filename from station name
            if hasattr( station, 'locationCode' ) and station.locationCode is not None:
                imgfile_name = '.'.join( ( station.networkCode, station.locationCode, station.stationCode, 'pickInfo' ) )
            else:
                imgfile_name = '.'.join( ( station.networkCode, station.stationCode, 'pickInfo' ) )

            
        if imgformat in ( 'eps', 'EPS' ):
            
            if imgfile is None:
                imgfile = os.path.join( metadata.pickInfoPlotDir, '.'.join( ( imgfile_name, 'eps' ) ) )
            
            epsfile = imgfile
            
        else:
            
            epsfile = os.path.join( metadata.pickInfoPlotDir, 'temp.eps' )

            if imgfile is None:
                imgfile = os.path.join( metadata.pickInfoPlotDir, '.'.join( ( imgfile_name, 'png' ) ) )

        show()
        savefig( epsfile )
        close()

        # convert to png
        if imgformat in ( 'png', 'PNG' ):
            os.system( 'convert -density 144 %s %s' % ( epsfile, imgfile ) )

            # remove EPS file
            os.remove( epsfile )

    def plotDistribution( self, station, metadata, imgfile = None, **kwargs ):
        """

        kwargs:
            imgformat: 'png' - convert to png, 'eps' no conversion
            colortable: has to be a colortable file in povray format
        """

        imgformat  = 'png'
        colortable = None

        if 'colortable' in kwargs.keys() and kwargs['colortable'] is not None:
            colortable = kwargs['colortable']
            
        if 'imgformat' in kwargs.keys() and kwargs['imgformat'] is not None:
            imgformat = kwargs['imgformat']
            
        ax = subplot(111)

        # load colortable
        # colordict = gmtColormap( colortable )

        cm = QPColorMap()
        cm.importPovray( colortable, flip=True, extend=True )

        my_cmap = cm.getMatplotlib()
        
        xysteps = station.distribution.distro_binning[self.distStyle-1]
        Xsteps  = xysteps[0]
        Ysteps  = xysteps[1]

        X1 = range( 0, int( (Xsteps[2]-Xsteps[0]) / Xsteps[1]+2 ), 1 )
        Y1 = range( 0, int( (Ysteps[2]-Ysteps[0]) / Ysteps[1]+2 ), 1 )

        X = numpy.zeros( ( len(X1), 1 ), dtype=float )
        for indx, value in enumerate(X1):
            X[indx] = ( value * Xsteps[1] ) + Xsteps[0] - Xsteps[1]/2

        Y = numpy.zeros( ( len(Y1), 1 ), dtype=float )
        for indx, value in enumerate(Y1):
            Y[indx]= ( value * Ysteps[1] ) + Ysteps[0] - Ysteps[1]/2

        x, y = meshgrid(X,Y)
        pcolor( x, y, station.distribution.distro, cmap = my_cmap, vmin = 0, vmax = 1 )
        colorbar()
        
        xlabel( 'Magnitude' )
        ylabel( 'Distance / km' )
        title( "Network " + station.networkCode + ", Station " + station.stationCode )
        axis( 'tight' )

        # get filename
        if imgfile is None:

            # get filename from station name
            if hasattr( station, 'locationCode' ) and station.locationCode is not None:
                imgfile_name = '.'.join( ( station.networkCode, station.locationCode, station.stationCode, 'distro' ) )
            else:
                imgfile_name = '.'.join( ( station.networkCode, station.stationCode, 'distro' ) )

            
        if imgformat in ( 'eps', 'EPS' ):
            
            if imgfile is None:
                imgfile = os.path.join( metadata.distroPlotDir, '.'.join( ( imgfile_name, 'eps' ) ) )
            
            epsfile = imgfile
            
        else:
            
            epsfile = os.path.join( metadata.distroPlotDir, 'temp.eps' )

            if imgfile is None:
                imgfile = os.path.join( metadata.distroPlotDir, '.'.join( ( imgfile_name, 'png' ) ) )

        show()
        savefig( epsfile )
        close()

        # convert to png
        if imgformat in ( 'png', 'PNG' ):
            os.system( 'convert -density 144 %s %s' % ( epsfile, imgfile ) )

            # remove EPS file
            os.remove( epsfile )

    def _eventDuringOntime( self, event, station ):

        # over channels / over ontime windows
        for curr_channel in station.channels:
            for curr_ontime in curr_channel.onTime:

                event_time = event.getPreferredOrigin().time.value

                if ( event_time >= curr_ontime[0] and event_time <= curr_ontime[1] ):
                    return True
        
        return False
    
    def _channelOperatingDuringPeriod( self, channel, timePeriod ):

        # timePeriod is list of [ startTime, endTime ] or None
        # startTime and endTime are of type QPDateTime
        
        # if channel has no start/end time, set as not operating
        if not ( ( channel.onTime is not None ) and len(channel.onTime) > 0 ):
            return False
        
        # if function is called without valid timePeriod, don't apply (return true)
        if timePeriod is None:
            return True
        else:
            startTime = QPDateTime( timePeriod[0] )
            endTime   = QPDateTime( timePeriod[1] )
            
        # if there is overlap between the two time windows, set channel as operating
        for curr_ontime in channel.onTime:
            if (    ( curr_ontime[0] <= startTime and curr_ontime[1] >= endTime )
                 or ( curr_ontime[0] >= startTime and curr_ontime[1] <= endTime )
                 or (     curr_ontime[0] <= startTime and curr_ontime[1] <= endTime
                      and curr_ontime[1] >= startTime )
                 or (     curr_ontime[0] >= startTime and curr_ontime[1] >= endTime
                      and curr_ontime[0] <= endTime ) ):
                return True
            
        # no overlap found
        return False
    
    def _eventPicked( self, event, network_code, station_code ):
      
        # over picks for given event
        for curr_pick in event.pick:
            
            # is current pick from station in question?
            if (     curr_pick.waveformID.stationCode == station_code 
                 and curr_pick.waveformID.networkCode == network_code ):
                return True
        
        return False
    
    def _deletePicks( self, event, network_code, station_code ):
      
        # over picks for given event
        for curr_pick_idx in reversed( xrange( len(event.pick) ) ):
            
            # if current pick is from station in question, delete pick and arrival
            if (     event.pick[curr_pick_idx].waveformID.stationCode == station_code 
                 and event.pick[curr_pick_idx].waveformID.networkCode == network_code ):
                
                # get arrival for current pick, delete
                curr_ori = event.getPreferredOrigin()
                for curr_arr_idx in reversed( xrange( len(curr_ori.arrival) ) ):
                    if curr_ori.arrival[curr_arr_idx].pickID == event.pick[curr_pick_idx]:
                        del curr_ori.arrival[curr_arr_idx]
                
                # delete pick
                del event.pick[curr_pick_idx]
                
    def _stationToBeUsed( self, station ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return True
    
    def _calcMagnitudeFromDistance( self, distance ):
        # this function has to be provided by the user
        # derive from PMC and fill in your code
        return 1.0
    # _calcMagnitudeFromDistance = Callable( _calcMagnitudeFromDistance )
    
    def _renameStation( self, station, newname ):
        station.stationCode = newname
        for ch in station.channels:
            ch.waveformID.stationCode = newname
            
    def _copyChannels( self, fromstation, tostation ):
        for ch in fromstation.channels:
            tostation.channels.append( ch )
    
    def mergeAliasStations( self ):
        
        # loop over alias list: networkCode | alias | stationCode
        for curr_alias in self.stationAliases:
        
            # look for alias in station list, can occur only once
            for curr_station_idx in reversed( xrange( len( self.stations ) ) ):

                curr_station = self.stations[curr_station_idx]
                
                # is alias matching entry in station list?
                # if not, do nothing, try next
                if (     curr_alias[0] == curr_station.networkCode
                     and curr_alias[1] == curr_station.stationCode ):
                    
                    # gotcha: look if we have already an entry for the original station name
                    originalFound = False
                    for check_station in self.stations:
                        if (     curr_alias[0] == check_station.networkCode
                             and curr_alias[2] == check_station.stationCode ):
                            
                            # main station name is there
                            # add alias station's channels to original station
                            self._copyChannels( curr_station, check_station )
                            
                            # delete entry for alias station
                            del self.stations[curr_station_idx]
                            originalFound = True

                            print " mergeAliases: copy data from station %s to %s" % ( curr_station.stationCode,
                                                                                       check_station.stationCode )
                            break
                    
                    # main station name is not there
                    # rename stationCode in station and channels
                    if not originalFound:
                        self._renameStation( curr_station, curr_alias[2] )

                        print " mergeAliases: rename station from %s to %s" % ( curr_station.stationCode,
                                                                                curr_alias[2] )
                            
                    # found alias, end station loop
                    break
                    
        return True

    def setDistroStyle( self, style ):

        if style != self.distStyle:
            self.distStyle = style

            for curr_sta in self.stations:
                curr_sta.distribution.restoreDistStyle( self.distStyle )
            
        return True
        
    def _getStationIdx( self, networkCode, stationCode ):
        for curr_sta_idx, curr_sta in enumerate(self.stations):
            if ( curr_sta.networkCode == networkCode and curr_sta.stationCode == stationCode ):
                return curr_sta_idx
        
        # if not yet returned, we have a new station
        self.stations.append( PMCStation( self.distStyle, smooth = self.distSmoothing ) )
        return len(self.stations)-1

    def selectStationsByNetworks( self, networks ):
        """
        use only stations that have a network code from networks
        networks is a list of network codes
        """

        # over stations in PMC
        for curr_sta_idx in reversed( xrange( len(self.stations) ) ):
            
            # if not in list of networks, delete
            if not self.stations[curr_sta_idx].networkCode in networks:
                del self.stations[curr_sta_idx]
                    
        return True

    def unselectStation( self, net_code, sta_code ):
        """
        unselect station with net_code, sta_code
        """

        # over stations in PMC
        for curr_sta_idx in reversed( xrange( len(self.stations) ) ):
            
            if (     ( self.stations[curr_sta_idx].networkCode == net_code )
                 and ( self.stations[curr_sta_idx].stationCode == sta_code ) ):
                del self.stations[curr_sta_idx]
                    
        return True
            
    def _preprocessStationScenario( self ):
        """
        has to be redefined in classes derived from PMC
        deselect stations from 'all' list for scenario computation
        """
        pass

    def _selectStationsInOperation( self, time ):
        """
        select stations by time
        time has to be mxDateTime object
        """

        curr_time = QPDateTime( time )
        
        # over stations
        for curr_sta_idx in reversed( xrange( len(self.stations) ) ):
            
            # over channels
            station_operating = False
            for curr_channel in self.stations[curr_sta_idx].channels:

                # over time intervals
                for curr_time_interval in curr_channel.onTime:

                    if ( curr_time >= curr_time_interval[0] and curr_time <= curr_time_interval[1] ):

                        # found an active channel
                        station_operating = True
                        break
                 
                if station_operating:
                    break

            # if no operating channel found, delete station
            if not station_operating:
                del self.stations[curr_sta_idx]
                
        return True

    def _computeStationDistances( self, location ):
        """
        compute array of station distances for one location
        input: location = ( lat, lon, depth )
        output: distances[] 
        """

        distances = []
        
        # over stations
        for curr_sta in self.stations:
            
            d, dummy = distanceBetweenPoints( location,
                                              ( curr_sta.latitude,
                                                curr_sta.longitude,
                                                -(curr_sta.elevation/1000.0) ),
                                              2 )
            # print d, dummy
            distances.append( d )
        return distances


    def _computeStationProbabilities( self, magnitude, distances ):
        """
        compute probability for a given magnitude and given distance vector
        """
        probabilities = numpy.zeros( ( len(self.stations), 2 ), dtype=float )
        prob_ctr = 0

        # over stations
        for curr_sta_idx, curr_sta in enumerate( self.stations ):

            curr_prob = curr_sta.distribution.getProbability( magnitude,
                                                              distances[curr_sta_idx],
                                                              self._calcMagnitudeFromDistance )
            # print curr_sta_idx, prob_ctr, curr_prob
            if curr_prob > 0.0:
                probabilities[prob_ctr, 0] = curr_prob
                probabilities[prob_ctr, 1] = 1.0 - curr_prob
                prob_ctr = prob_ctr+1

        probabilities.resize( ( prob_ctr, 2 ) )
        return probabilities

    def _computeCombinedProbabilities( self, probabilities ):
        """
        compute combined probability for a vector of station probabilities
        input parameter probabilities has to be numpy array
        """
        numUsedStations = probabilities.shape[0]
        #print " numUsedStations = %s" % numUsedStations

        if numUsedStations < self.stationsRequiredForTriggering:
            return 0.0

        prod_0 = probabilities[:,1].prod()

        # compute prod_1
        prod_1 = 0.0
        for idx in xrange( 0, numUsedStations ):
            p = probabilities[:,1].copy()
            p[idx] = probabilities[idx, 0]
            prod_1 = prod_1 + p.prod()
            
        prod = numpy.zeros( self.stationsRequiredForTriggering, dtype=float )

        for num in xrange( 2, self.stationsRequiredForTriggering, 1 ):

            # get combinations list for num (m-tuple of combinations) and numUsedStations
            # get starting index of full list in combinations we have to use:
            # combinations.index[num][numUsedStations]
            # slice ends at end of full list
            
            # use generator method getCombinations
            # combinations have to be pre-computed up to self.stationsRequiredForTriggering-2
            # i.e. for Southern California: no of triggering stations is 4, need pre-computed combinations up to 2
            # i.e. for Northern California: no of triggering stations is 5, need pre-computed combinations up to 3
            for combination in self.combinations.getCombinations( num, numUsedStations ):

                #print " dim of single combination: %s, %s" % ( len( combination ), combination )
                
                # make copy of "anti"-probabilities (1-P)
                p = probabilities[:,1].copy()

                # dimension of combination is num
                # replace 1-P with P in array p at indices given in combination
                for idx in combination:
                    p[idx] = probabilities[idx, 0]

                prod[num] = prod[num] + p.prod()

        # return overall probability
        return 1 - prod_0 - prod_1 - prod.sum()
    
    def getProbability( self, location, magnitude, time, maxStations ):
        """
        get combined probability for (location, magnitude, time)

        1) preprocess station scenario
        2) get distance vector for given location
        3) get probability vector for given magnitude and distance vector
        4) do combinatorics: get combined probability from probability vector
        """

        self._preprocessStationScenario()
        self._selectStationsInOperation( time )

        distances     = self._computeStationDistances( location )
        probabilities = self._computeStationProbabilities( magnitude, distances )

        # sort probabilities (largest first)
        # probabilities is numpy array [ prob, 1-prob ]
        # sort both columns
        indices = probabilities[:,0].argsort(axis=0)
        sortedProbabilities = probabilities[numpy.flipud(indices)]

        # if first (requiredForTriggering) probabilities are 1.0, return 1.0
        # otherwise: compute combined probability
        if floatEqual( sortedProbabilities[(self.stationsRequiredForTriggering-1), 0], 1.0 ):
            return 1.0
        else:
            combinedProbability = self._computeCombinedProbabilities( sortedProbabilities[0:(maxStations-1),:] )
            return combinedProbability

    def _getProbabilitySet( self, location, time, magnitudes, maxStations ):
        """
        get probability array for (location, time) and a list of magnitudes
        does not compute station scenario beforehand
        
        input:  list of magnitudes
        output: list of [ sorted magnitude | combined probability ]
        """

        distances = self._computeStationDistances( location )

        combinedProbabilities = []
        probThresholdReached  = False
        
        for mag in sorted( magnitudes ):
            
            if probThresholdReached is True:
                combinedProbabilities.append( [ mag, 1.0 ] )
                continue
                
            else:
                probabilities = self._computeStationProbabilities( mag, distances )

                # if numpy array is empty (all probabilities are 0), return 0.0
                if probabilities.shape[0] < self.stationsRequiredForTriggering:
                    combinedProbabilities.append( [ mag, 0.0 ] )
                    continue
                else:
                
                    # sort probabilities (largest first)
                    # probabilities is numpy array [ prob, 1-prob ]
                    # sort both columns
                    indices = probabilities[:,0].argsort(axis=0)
                    sortedProbabilities = probabilities[numpy.flipud(indices)]

                    # if first (requiredForTriggering) probabilities are 1.0, return 1.0
                    # otherwise: compute combined probability
                    if floatEqual( sortedProbabilities[(self.stationsRequiredForTriggering-1), 0], 1.0 ):
                        probThresholdReached = True
                        combinedProbabilities.append( [ mag, 1.0 ] )
                        
                    else:
                        combinedProbability = self._computeCombinedProbabilities( sortedProbabilities[0:(maxStations-1),:] )

                        if combinedProbability >= self.probThreshold:

                            # set all further probabilities to 1.0
                            probThresholdReached = True
                            combinedProbabilities.append( [ mag, 1.0 ] )
                            continue

                        else:
                            combinedProbabilities.append( [ mag, combinedProbability ] )

        return combinedProbabilities
                
    def getProbabilitySet( self, location, time, magnitudes ):
        """
        get probability array for (location, time) and a list of magnitudes
        compute station scenario beforehand
        
        input:  list of magnitudes
        output: list of [ sorted magnitude | combined probability ]
        """

        self._preprocessStationScenario()
        self._selectStationsInOperation( time )

        return self._getProbabilitySet( location, time, magnitudes )

    def getProbabilityMap( self, grid, metadata, **kwargs ):
        """
        compute PMCData on grid nodes for given sets of magnitudes and probabilities
        computes station scenario beforehand

        adds PMCData nodes to input QPGrid

        input:
            grid         - PMCGridN on which PMC data is computed
            metadata     - metadata for computation run

             used from metadata:
               .targetMagArray  - list of magnitudes for which probabilities are computed
               .targetProbArray - list of probabilities for which completeness magnitudes are computed
               .useDate         - mx.DateTime object, point in time for which PMC computation is made
        
        kwargs: dumpfile - dump grid in ASCII format, lon | lat | probability
                           one file for each magnitude
                depth    - use this depth instead of value in depthLayer
        """

        self._preprocessStationScenario()
        self._selectStationsInOperation( metadata.useDate )
        
        if ( 'dumpfile' in kwargs.keys() ) and ( kwargs['dumpfile'] is not None ):
            dumpfilename = kwargs['dumpfile']
            dumpFile = True
        else:
            dumpFile = False 

        if ( 'depth' in kwargs.keys() ) and ( kwargs['depth'] is not None ):
            depth = float( kwargs['depth'] )
        else:
            depth = None

        asciimap = []
        for mag_idx in xrange( len( metadata.targetMagArray ) ):
            asciimap.append( [] )

        # over QPGrid depth layers
        for curr_depthlayer_idx in xrange( grid.depthLayer.shape[0] ):

            if depth is not None:
                curr_depth = depth
            else:
                curr_depth = grid.depthLayer[curr_depthlayer_idx,0]

            # TODO: select cells for current depth layer
            for curr_cell_idx in xrange( grid.cells.shape[0] ):

                curr_lon = grid.cells[curr_cell_idx,0]
                curr_lat = grid.cells[curr_cell_idx,1]

                curr_location = ( curr_lat, curr_lon, curr_depth )

                if ( 'verbose' in kwargs.keys() ) and ( kwargs['verbose'] is True ):
                    print " compute cell: lon = %s, lat = %s, depth = %s" % ( curr_lon, curr_lat, curr_depth )

                # compute array of probabilities for target magnitudes
                curr_mag_prob_list = self._getProbabilitySet( curr_location,
                                                              metadata.useDate,
                                                              metadata.targetMagArray,
                                                              metadata.useMaxStationCnt )

                # loop over computed probabilities and write to pmcDataProb array
                for curr_prob_idx, curr_prob_val in enumerate( curr_mag_prob_list ):

                    grid.pmcDataProb[curr_cell_idx,curr_prob_idx] = curr_prob_val[1]

                    if dumpFile:
                        asciimap[curr_prob_idx].append( '\t'.join( ( "%6.2f" % curr_lon,
                                                                     "%6.2f" % curr_lat,
                                                                     "%6.4f" % curr_prob_val[1] ) ) )

                if metadata.targetProbArray is not None:

                    # loop over target probabilities, compute Mp value, and write to
                    # pmcDataMP array
                    for mpp_idx, mpp in enumerate( metadata.targetProbArray ):

                        grid.pmcDataMP[curr_cell_idx,mpp_idx] = self.getPMC( mpp, curr_mag_prob_list )

        if dumpFile:        
            for mag_idx in xrange( len(asciimap) ):

                filename = str(metadata.targetMagArray[mag_idx]) + '.' + dumpfilename
                fh = writeQPData( filename, **kwargs )
                try:
                    fh.write( '\n'.join( asciimap[mag_idx] ) )
                except:
                    raise IOError, "PMC::writexml - error in toxml()"
                fh.close()
            
        return True

    def getPMC( self, targetProbability, magProbList ):
        """
        input: [ mag_array, prob_arr ], target probability
        output: magnitude of completeness
        """
        for curr_mag_prob in magProbList:
            if targetProbability <= curr_mag_prob[1]:
                return curr_mag_prob[0]
        return numpy.nan

    # getMagnitudeOfCompleteness()
    # calls getProbabilityArray
    #       getPMC()
    def getMagnitudeOfCompleteness( self ):
        pass
