#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

#
# jma_compGrid.py
# $Id: jma_compGrid.py 200 2009-04-30 10:46:47Z fab $
#

############################################################################
#    Copyright (C) 2008-2009 by Fabian Euchner and Danijel Schorlemmer     #
#    fabian@fabian-euchner.de                                              #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

import sys
import os
import getopt
import glob
import re
import random
import datetime

import pickle
import cPickle

from mx.DateTime     import Date
from mx.DateTime     import DateTime
from mx.DateTime.ISO import ParseDateTimeUTC

sys.path.append('../../..')
sys.path.append('../..')
sys.path.append('..')

from QPCore    import *
from QPUtils   import *
from QPCatalog import *
from QPPolygon import *

from pmc.PMC         import *
from pmc.PMC_JMA     import *
from pmc.PMCMetadata import *

from pmc.PMCGridN    import *

# metadata for this computation run
metadata = PMCMetadata()

## set a few paths that do not go into metadata

# this gets the absolute base path  == path where this script resides
# subsequent directory hierarchy will be below this directory
scriptname = os.path.basename( os.path.abspath( sys.argv[0] ) )
basepath   = os.path.dirname( os.path.abspath( sys.argv[0] ) ) # no trailing slash

# directory where data from this run will be written to
rundir_base = os.path.join( basepath, 'data/runs/' )

# location of station list and station alias files
stationfile_path = os.path.join( basepath, 'data/station/' )

# default station file name
default_stationfile_name = 'jma_stations_districts-2009-03-05.dat'

# default alias file name
default_aliasfile_name = None

# default polygon for Japan
default_polygon = ( ( 120.0, 20.0 ),
                    ( 120.0, 36.0 ),
                    ( 134.0, 49.0 ),
                    ( 150.0, 49.0 ),
                    ( 150.0, 20.0 ),
                    ( 120.0, 20.0 ) )

# location of combinations pickle
combi_path = os.path.join( basepath, 'data/combinations' )

## set metadata

# directory where catalog (deck) files in bz2 format reside
# deck file names must end with '.Z.bz2'
# works only for deck files with A,B,C part in names (starting from October 1997)
metadata.catalogBaseDir      = os.path.join( basepath, 'data/catalog-bz2/' )
metadata.catalogFilesPattern = '*.Z.bz2'

# distStyle for prob distros
metadata.useDistStyle = 5
metadata.smoothDist   = True

# time zone difference to UTC in hours
# metadata.timeZoneShift = 9.0

# PMC grid file prefix
metadata.gridFilePrefix = 'jma'

# metadata file prefix
metadata_file_prefix = 'pmcinfo'

# combinations pickle file
metadata.combinationsPickleFile = os.path.join( combi_path, 'combinations-3-200.numpy.pickle' )

# color map file for prob distros
metadata.probDistroColorMapFile = os.path.join( basepath, '../../data/colormaps/pov/rastafari.pov' )

# ---------------------------------------------------------------------------

# verbosity level
verbose_level = 2
verbose_grid  = True

# ---------------------------------------------------------------------------

# stuff for resulting grid XML
metadata.targetMagArray  = frange( -1.0, 5.0, 0.1 )    # -1.0, -5.0, 0.1
metadata.targetProbArray = [ 0.9, 0.95, 0.99, 0.999 ]

# ---------------------------------------------------------------------------

# coordinate limits: lon (120,150), lat (20,49)

metadata.areaPolygon = { 'vertices': default_polygon,
                         'metadata': { 'lonDelta'              : 0.1,
                                       'latDelta'              : 0.1,
                                       'lonAlign'              : 0.0,
                                       'latAlign'              : 0.0,
                                       'includePointOnBoundary': True,
                                       'shift'                 : False  }
                       } 

metadata.areaDepth = 30.0


# ---------------------------------------------------------------------------

def main():

    global scriptname
    global metadata

    global verbose_level
    
    setUp()

    if (     ( metadata.combinePickInfoProbDistro is False )
         and ( metadata.skipProbDistro is True )
         and (     metadata.skipPickInfo is False
               and metadata.skipGrid is False ) ):
        print "%s error: skipping only distro creation (-D) does not make sense. Combine with switches -P and -G" % ( scriptname )
        sys.exit()

    if (     ( metadata.discardPickInfo is True )
         and ( metadata.combinePickInfoProbDistro is False ) ):
        print "%s error: you cannot discard pickInfo if not combined with distro computation. Use switch -C" % ( scriptname )
        sys.exit()

    if metadata.skipPickInfo is False:
        compPickInfos( verbose_level = verbose_level )

    if metadata.skipProbDistro is False:
        compProbDistros( verbose_level = verbose_level )

    if metadata.skipGrid is False:
        compGrids( verbose_level = verbose_level )

# ---------------------------------------------------------------------------

def setUp():

    global rundir_base
    global scriptname

    global metadata
    
    global stationfile_path
    global metadata_file_prefix

    # command line variables
    in_skipdistro       = False
    in_skipgrid         = False
    in_skippick         = False
    in_combipickprob    = False
    in_discardpickinfo  = False

    in_plotpickinfo     = False
    in_plotdistro       = False
    
    in_aliasfile_name   = None
    in_polygonfile_name = None
    
    in_date             = None
    in_griddir          = None
    in_stationfile_name = None
    in_earlier          = None
    in_later            = None
    in_workingdir       = None

    in_max_stations_cnt = 100
               
    # Read commandline arguments
    cmdParams = sys.argv[1:]
    if len( cmdParams ) == 0:
        PrintHelp()
        sys.exit()
            
    opts, args = getopt.gnu_getopt( cmdParams,
                                    'hDGPCXfFa:d:g:N:p:s:t:T:w:',
                                    [ 'help', 'skipprobdistro', 'skipgrid',
                                      'maxstations', 'skippickinfo',
                                      'combipickprob', 'discardpickinfo',
                                      'plotpickinfo', 'plotdistro',
                                      'aliasfile=', 'polygonfile=', 'date=', 'griddir=',
                                      'stationfile=', 'timebefore=', 'timeafter=',
                                      'workingdir=' ] )

    for option, parameter in opts:

        if option == '-D' or option == '--skipprobdistro':
            in_skipdistro = True

        if option == '-G' or option == '--skipgrid':
            in_skipgrid = True
            
        if option == '-P' or option == '--skippickinfo':
            in_skippick = True

        if option == '-C' or option == '--combipickprob':
            in_combipickprob = True

        if option == '-X' or option == '--discardpickinfo':
            in_discardpickinfo = True

        if option == '-f' or option == '--plotpickinfo':
            in_plotpickinfo = True

        if option == '-F' or option == '--plotdistro':
            in_plotdistro = True
            
        if option == '-a' or option == '--aliasfile':
            in_aliasfile_name = parameter

        if option == '-p' or option == '--polygonfile':
            in_polygonfile_name = parameter
            
        if option == '-d' or option == '--date':
            in_date = parameter

        if option == '-g' or option == '--griddir':
            in_griddir = parameter

        if option == '-N' or option == '--maxstations':
            in_max_stations_cnt = parameter
            
        if option == '-s' or option == '--stationfile':
            in_stationfile_name = parameter

        if option == '-t' or option == '--timebefore':
            in_earlier = parameter

        if option == '-T' or option == '--timeafter':
            in_later = parameter

        if option == '-w' or option == '--workingdir':
            in_workingdir = parameter

        if option == '-h' or option == '--help':
            PrintHelp()
            sys.exit()

    try:
        # useDate as mxDateTime instance
        metadata.useDate     = ParseDateTimeUTC( in_date )
        metadata.earlierDays = int( in_earlier )
        metadata.laterDays   = int( in_later )
    except:
        error_str = "%s - illegal date format %s or non-integer time interval" % ( scriptname, in_date )
        raise ValueError, error_str

    # get start and end dates
    metadata.startDate = metadata.useDate - DateTimeDelta( float( metadata.earlierDays ) )
    metadata.endDate   = metadata.useDate + DateTimeDelta( float( metadata.laterDays ) )

    # max stations used for combined probability
    metadata.useMaxStationCnt = in_max_stations_cnt

    # set working dircetories
    if in_workingdir is not None:

        # working directory has explicitly been given
        metadata.runPath = os.path.join( rundir_base, in_workingdir )

        if not os.path.isdir( metadata.runPath ):
            os.makedirs( metadata.runPath )

        # if grid dir is given, create directly below output dir (one level above pickInfo and distro)
        # no date component in dir name
        # this is to collect all grid files of a sequence of runs
        # if no explicit grid dir is given, create std grid dir below working dir
        if in_griddir is not None:
            metadata.gridDir = os.path.join( metadata.runPath, in_griddir )
            
            if not os.path.isdir( metadata.gridDir ):
                os.makedirs( metadata.gridDir )

        # we know where to find pickInfo and probdistro
        # i.e., skipping of computation is possible
        # set flags for skipping computation
        metadata.skipPickInfo              = in_skippick
        metadata.skipProbDistro            = in_skipdistro
        metadata.skipGrid                  = in_skipgrid

    else:

        # no explicit working directory, use PID
        metadata.runPath = os.path.join( rundir_base, str( os.getpid() ) )

        if os.path.isdir( metadata.runPath ):
            error_str = "%s - output path already exists, %s" % ( scriptname, metadata.runPath )
            raise IOError, error_str
        else:
            os.makedirs( metadata.runPath )

    # skipping/combining options
    metadata.combinePickInfoProbDistro = in_combipickprob
    metadata.discardPickInfo           = in_discardpickinfo

    # setting combined pickInfo and prob distro switch overrides individual skip switches
    if metadata.combinePickInfoProbDistro is True:
        metadata.skipPickInfo   = False
        metadata.skipProbDistro = True
            
    # plotting of results
    metadata.plotPickInfo   = in_plotpickinfo
    metadata.plotProbDistro = in_plotdistro

    # set instance dir: runPath + date
    instanceDir = os.path.join( metadata.runPath,
                                mxDateTime2ISO( metadata.useDate, showtime = False ) )
     
    # if new pickInfos or distros are computed:
    # if instance path exists, stop program execution (don't overwrite)
    if (     os.path.isdir( instanceDir )
         and ( metadata.skipPickInfo is False or metadata.skipProbDistro is False ) ):
        error_str = "%s - instance directory already exists, %s" % ( scriptname, instanceDir )
        raise IOError, error_str
        
    # create pickInfo and distro dirs
    metadata.pickInfoDir = os.path.join( instanceDir, 'pickInfo/' )
    metadata.distroDir   = os.path.join( instanceDir, 'distro/' )

    metadata.pickInfoPlotDir = os.path.join( instanceDir, 'pickInfo-plot/' )
    metadata.distroPlotDir   = os.path.join( instanceDir, 'distro-plot/' )
                                         
    # create grid dir, if not explicitly set
    if metadata.gridDir is None:
        metadata.gridDir = os.path.join( instanceDir, 'grid/' )
        
    # if working directory was explicitly set, directories may already exist
    if not os.path.isdir( metadata.pickInfoDir ):
        os.makedirs( metadata.pickInfoDir )

    if not os.path.isdir( metadata.distroDir ):
        os.makedirs( metadata.distroDir )

    if not os.path.isdir( metadata.gridDir ):
        os.makedirs( metadata.gridDir )

    if not os.path.isdir( metadata.pickInfoPlotDir ):
        os.makedirs( metadata.pickInfoPlotDir )

    if not os.path.isdir( metadata.distroPlotDir ):
        os.makedirs( metadata.distroPlotDir )

    if in_stationfile_name is not None:
        metadata.stationfile = os.path.join( stationfile_path, in_stationfile_name )
    else:
        metadata.stationfile = os.path.join( stationfile_path, default_stationfile_name )

    if in_aliasfile_name is not None:
        metadata.aliasfile = os.path.join( stationfile_path, in_aliasfile_name )
    elif default_aliasfile_name is not None:
        metadata.aliasfile = os.path.join( stationfile_path, default_aliasfile_name )

    if in_polygonfile_name is not None:
        metadata.polygonfile = os.path.join( stationfile_path, in_polygonfile_name )
        
    # set metadata file
    metadataFilename = '.'.join( ( metadata_file_prefix,
                                   mxDateTime2ISO( metadata.useDate, showtime = False ),
                                   '00.dat' ) )
    metadata.metadataFile = os.path.join( metadata.runPath, metadataFilename )
    
    # check if metadata file exists, if yes, try another name
    metadataFileCtr = 0
    while ( os.path.isfile( metadata.metadataFile ) ):

        print " skipping metadata file %s" % metadata.metadataFile

        metadataFileCtr += 1
        metadataFilename = '.'.join( ( metadata_file_prefix,
                                       mxDateTime2ISO( metadata.useDate, showtime = False ),
                                       "%02d" % metadataFileCtr,
                                       'dat' ) )
        metadata.metadataFile = os.path.join( metadata.runPath, metadataFilename )
    
    # write metadata to file and screen
    print " write run metadata to file %s" % metadata.metadataFile
    writeMetadata( writeQPData( metadata.metadataFile ) )
    writeMetadata( sys.stdout )


def writeMetadata( stream ):
    """
    metadata is a dictionary with relevant metadata of computation run
    """

    global metadata
    
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )
    stream.write( ''.join( ( "PMC/JMA computation for date %s" % \
                  mxDateTime2ISO( metadata.useDate, showtime = False ), '\n' ) ) )
    stream.write( ''.join( ( " phase data starts at %s (%s days earlier)" % \
                  ( mxDateTime2ISO( metadata.startDate, showtime = False ),
                    metadata.earlierDays ), '\n' ) ) )
    stream.write( ''.join( ( " phase data ends at %s (%s days later)" % \
                  ( mxDateTime2ISO( metadata.endDate, showtime = False ),
                    metadata.laterDays ), '\n' ) ) )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )

    if metadata.combinePickInfoProbDistro is True:
        stream.write( ''.join( ( " COMPUTING combined pickInfo and probability distributions", '\n' ) ) )

        if metadata.discardPickInfo is True:
            stream.write( ''.join( ( " -> DISCARDING pickInfo files", '\n' ) ) )
    else:
        if metadata.skipPickInfo is True:
            stream.write( ''.join( ( " SKIPPING pickInfo", '\n' ) ) )
        else:
            stream.write( ''.join( ( " COMPUTING pickInfo", '\n' ) ) )
        if metadata.skipProbDistro is True:
            stream.write( ''.join( ( " SKIPPING probability distributions", '\n' ) ) )
        else:
            stream.write( ''.join( ( " COMPUTING probability distributions", '\n' ) ) )

    if metadata.skipGrid is True:
        stream.write( ''.join( ( " SKIPPING PMC Grid", '\n' ) ) )
    else:
        stream.write( ''.join( ( " COMPUTING PMC Grid", '\n' ) ) )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )
    stream.write( ''.join( ( " input files:", '\n' ) ) )
    stream.write( ''.join( ( "  station file: %s" % metadata.stationfile, '\n' ) ) )
    stream.write( ''.join( ( "  alias file: %s" % metadata.aliasfile, '\n' ) ) )
    stream.write( ''.join( ( "  polygon file: %s" % metadata.polygonfile, '\n' ) ) )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )
    stream.write( ''.join( ( " operating on paths:", '\n' ) ) )
    stream.write( ''.join( ( "  pickInfo dir: %s" % metadata.pickInfoDir, '\n' ) ) )
    stream.write( ''.join( ( "  prob distro dir: %s" % metadata.distroDir, '\n' ) ) )
    stream.write( ''.join( ( "  grid dir: %s" % metadata.gridDir, '\n' ) ) )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )
    stream.write( ''.join( ( " computation parameters:", '\n' ) ) )
    stream.write( ''.join( ( "  max. number of stations for combined probabilities: %s" % metadata.useMaxStationCnt, '\n' ) ) )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )
    stream.write( ''.join( ( "the command line was:", '\n' ) ) )
    stream.write( ' '.join( sys.argv ) )
    stream.write( '\n' )
    stream.write( ''.join( ( "=========================================================================================", '\n' ) ) )


def compPickInfos( **kwargs ):

    global metadata
    
    print " ========== computing pickInfos =========="

    useDateStr = mxDateTime2ISO( metadata.useDate, showtime = False )
    
    # new inventory object for useDate

    # create output dir
    if not os.path.isdir( metadata.pickInfoDir ):
        print "   creating path %s" % metadata.pickInfoDir
        os.makedirs( metadata.pickInfoDir )

    if metadata.combinePickInfoProbDistro is True:
        if not os.path.isdir( metadata.distroDir ):
            print "   creating path %s" % metadata.distroDir
            os.makedirs( metadata.distroDir )

    print " importing PMC station file: %s" % metadata.stationfile
    pmc = PMC_JMA( metadata.stationfile, 0, encoding = 'html' )

    # set timeZoneShift in metadata
    metadata.timeZoneShift = pmc.timeZoneShift
    
    if metadata.aliasfile is not None:
        print " importing PMC alias file: %s" % metadata.aliasfile
        pmc.importAliases( metadata.aliasfile )

    print " before preprocess: PMC inventory has %s stations" % len( pmc.stations )
    pmc.preprocessInventory( [ metadata.startDate, metadata.endDate ] )
    print " after inventory preprocess: inventory has %s stations" %  len( pmc.stations )

    metadata.getCatalogFiles()

    # loop over catalog files
    print " ----- assign picks for %s stations, using %s catalog files" % ( len( pmc.stations ),
                                                                            len( metadata.catalogFiles ) )
    pmc.assignPicks( metadata, compression='bz2', verbose_level = kwargs['verbose_level'] )
    
    # pmc.mergeAliasStations()
    # print " after merging aliases: inventory has %s stations" %  len( pmc.stations )
    
def compProbDistros( **kwargs ):

    global metadata

    print " ========== computing probability distros =========="

    # create output dir
    if not os.path.isdir( metadata.distroDir ):
        print "   creating path %s" % metadata.distroDir
        os.makedirs( metadata.distroDir )

    # get pick info files
    pickinfofiles = sorted( glob.glob( os.path.join( metadata.pickInfoDir, '*.pickInfo.xml.bz2' ) ) )
    
    # loop over stations
    for pickinfofile in pickinfofiles:

        print " processing pick info file %s" % ( pickinfofile )

        # create PMC for current station
        curr_pmc = PMC_JMA()
        curr_pmc.readXML( getQPDataSource( pickinfofile, compression='bz2' ) )

        # fillup
        curr_pmc.stations[0].distribution.restoreDistStyle( metadata.useDistStyle )
        curr_pmc.stations[0].distribution.setSmooth( metadata.smoothDist )

        print " ----- now calling fillup ..."
        curr_pmc.stations[0].fillup( curr_pmc._calcMagnitudeFromDistance )

        # delete pick info
        if hasattr( curr_pmc.stations[0].distribution, 'pickInfo' ):
            del curr_pmc.stations[0].distribution.pickInfo

        # write distro to disk
        curr_pmc.writeProbDistros( metadata, verbose_level = kwargs['verbose_level']  )

        # plot prob distro
        if metadata.plotProbDistro is True:

            if verbose_level is not None and ( verbose_level > 1 ):
                print " --- creating plot of prob distro"
                            
            curr_pmc.plotDistribution( curr_pmc.stations[0], metadata,
                                       colortable = metadata.probDistroColorMapFile )
        
        # delete station just processed
        del curr_pmc
        
def compGrids( **kwargs ):
    """
    Note: output grid file is gzipped
          do not use built-in pretty-print (uses too much memory)
          use 'xmllint --format' for pretty-print
          has name <prefix>.YYYY-MM-DDTHH-MM-SS.xml.gz
    """

    global metadata
    global verbose_grid

    print " ========== computing JMA PMC grid =========="
    
    print " ----- get PMC from XML -----  "

    # create output dir
    if not os.path.isdir( metadata.gridDir ):
        print "   creating path %s" % metadata.gridDir
        os.makedirs( metadata.gridDir )
        
    print " load Combinations from pickle"
    combi = unpickleObj( metadata.combinationsPickleFile )
        
    pmc = PMC_JMA( combinations = combi )

    # import PMC distro data: loop over distro files for stations
    distrofiles = sorted( glob.glob( os.path.join( metadata.distroDir, '*.distro.xml.bz2' ) ) )

    for curr_sta_file in distrofiles:

        print " importing PMC from XML file: %s" % curr_sta_file

        curr_pmc = PMC_JMA()
        curr_pmc.readXML( getQPDataSource( curr_sta_file, compression='bz2' ) )

        # copy over PMC for station to master PMC object
        pmc.stations.append( curr_pmc.stations[0] )

        del curr_pmc

    print " loaded PMC from XML: %s stations" % ( len( pmc.stations ) )

    print " ----- compute QPGrid -----  "

    ## compute grid

    # create polygon object for border of region
    if metadata.polygonfile is None:
        polygon = QPPolygon( metadata.areaPolygon['vertices'] )
    else:
        polygon = QPPolygon( metadata.polygonfile )
        metadata.areaPolygon['vertices'] = polygon.vertices
    
    g = PMCGridN( metadata )
    g.setGridParameter( metadata.areaPolygon['metadata'] )
    g.setupPolygon( polygon, metadata )

    # add annotation object to grid
    g.annotation = pmc.annotation

    g.annotation.setProperty( date                 = utc(),
                              starttime            = metadata.useDate,
                              endtime              = metadata.useDate,
                              lonmin               = g.extent['lonMin'],
                              lonmax               = g.extent['lonMax'],
                              latmin               = g.extent['latMin'],
                              latmax               = g.extent['latMax'],
                              boundary             = polygon.vertices,
                              observationStartTime = metadata.startDate,
                              observationEndTime   = metadata.endDate,
                              maxStationCnt        = metadata.useMaxStationCnt )
                                
    print " compute probability map for Date: %s" % ( metadata.useDate )
    pmc.getProbabilityMap( g, metadata, verbose = verbose_grid )

    # add station data to grid - delete not required fields
    g.stations = []
    for curr_sta in pmc.stations:
        del curr_sta.channels
        del curr_sta.distribution
        g.stations.append( curr_sta )

    mapdate_str = mxDateTime2ISO( [ metadata.useDate ], timesepreplacechar = '-', showsecfrac = False )

    grid_file_temp = os.path.join( metadata.gridDir, '.'.join( ( metadata.gridFilePrefix, mapdate_str, 'short.xml' ) ) )
    grid_file_out  = os.path.join( metadata.gridDir, '.'.join( ( metadata.gridFilePrefix, mapdate_str, 'xml.gz' ) ) )

    print " write uncompressed, compact grid file %s" % grid_file_temp
    g.writeXML( grid_file_temp, prettyPrint = False )

    print " creating pretty-printed, compressed grid file: %s" % grid_file_out
    exec_str = "xmllint --format %s | gzip > %s" % ( grid_file_temp, grid_file_out )
    os.system( exec_str )

    # remove compact file
    os.remove( grid_file_temp )

def PrintHelp():
    global scriptname
    
    print 'computes PMC grid'
    print 'Usage: %s [OPTION]' % scriptname
    print '  Options'
    print '   -a FILE, --aliasfile=<file>      Alias file name'
    print '   -C, --combipickprob              Compute prob distro directly after pickInfo for each station'
    print '   -d VALUE, --date=<value>         Date for which PMC grid is computed (YYYY-MM-DD)'
    print '   -D, --skipprobdistro             Skip computation of probability distributions'
    print '   -f, --plotpickinfo               Plot pickInfos'
    print '   -F, --plotdistro                 Plot probability distros'
    print '   -g VALUE, --griddir=<value>      Directory to place PMC grid file'
    print '   -G, --skipgrid                   Skip computation of PMC Grid'
    print '   -N, --maxstations                Maximum number of stations used for probability computation (default: 100)'
    print '   -P, --skippickinfo               Skip computation of pickInfo'
    print '   -p FILE, --polygonfile=<file>    Polygon boundary file name'
    print '   -s FILE, --stationfile=<file>    Station file name'
    print '   -t VALUE, --timebefore=<value>   Days before date'
    print '   -T VALUE, --timeafter=<value>    Days after date'
    print '   -w VALUE, --workingdir=<value>   Working directory'
    print '   -X, --discardpickinfo            Do not store pickInfo files, only together with -C option'
    print '   -h, --help                       print this information'

if __name__ == "__main__":
    main()
